from typing import List, Union, Optional, Literal, Dict, Any
from pydantic import BaseModel, root_validator
from pydantic import model_validator

# ===== 基础结构类型 =====

class Title(BaseModel):
    text: Optional[str] = None
    left: Optional[str] = None
    top: Optional[str] = None


class Tooltip(BaseModel):
    trigger: Optional[str] = None
    axisPointer: Optional[Dict[str, Any]] = None


class Legend(BaseModel):
    orient: Optional[str] = None
    left: Optional[str] = None
    data: Optional[List[str]] = None


class Grid(BaseModel):
    left: Optional[Union[str, int]] = None
    right: Optional[Union[str, int]] = None
    top: Optional[Union[str, int]] = None
    bottom: Optional[Union[str, int]] = None
    containLabel: Optional[bool] = None


class XAxis(BaseModel):
    type: Optional[str] = None  # category, value, time, log
    data: Optional[List[Union[str, int, float]]] = None


class YAxis(BaseModel):
    type: Optional[str] = None


class Radar(BaseModel):
    indicator: Optional[List[Dict[str, Union[str, int]]]] = None
    shape: Optional[str] = None


# ===== Series 类型（通用） =====

class BaseSeries(BaseModel):
    type: Literal['bar', 'line', 'pie', 'scatter', 'radar', 'gauge', 'heatmap']
    name: Optional[str] = None
    data: Optional[List[Union[int, float, Dict[str, Any]]]] = None
    smooth: Optional[bool] = None
    stack: Optional[str] = None
    radius: Optional[Union[str, List[str]]] = None   # for pie
    center: Optional[List[str]] = None               # for pie
    roseType: Optional[str] = None                   # for pie
    itemStyle: Optional[Dict[str, Any]] = None
    lineStyle: Optional[Dict[str, Any]] = None
    areaStyle: Optional[Dict[str, Any]] = None
    label: Optional[Dict[str, Any]] = None
    emphasis: Optional[Dict[str, Any]] = None
    min: Optional[float] = None                      # for gauge
    max: Optional[float] = None                      # for gauge
    pointer: Optional[Dict[str, Any]] = None         # for gauge


# ===== 最终 Option 配置结构 =====

class EChartsOption(BaseModel):
    title: Optional[Title] = None
    tooltip: Optional[Tooltip] = None
    legend: Optional[Legend] = None
    grid: Optional[Union[Grid, List[Grid]]] = None
    xAxis: Optional[Union[XAxis, List[XAxis]]] = None
    yAxis: Optional[Union[YAxis, List[YAxis]]] = None
    radar: Optional[Radar] = None
    series: List[BaseSeries]
    color: Optional[List[str]] = None
    backgroundColor: Optional[Union[str, Dict[str, Any]]] = None
    animation: Optional[bool] = None
    textStyle: Optional[Dict[str, Any]] = None
    toolbox: Optional[Dict[str, Any]] = None
    dataset: Optional[Union[Dict[str, Any], List[Dict[str, Any]]]] = None

    class Config:
        extra = "allow"  # 允许传递未声明字段

    @model_validator(mode='after')
    def check_series_exists(cls, values):
        if not values.series or len(values.series) == 0:
            raise ValueError("配置项中必须包含至少一个 series。")
        return values
