import { ref, nextTick } from 'vue';
import useToolCallStore from "@/pages/chat/composables/useToolCallStore.js";
import {reactive} from "@vue/runtime-dom";

const messages = ref([]);
const status = ref('idle');
let ws = null;

// 用来记录流消息，key是stream_id，value是消息对象
const streamMap = new Map();
const pendingToolCallIds = ref(new Set());
const {  setSelectedToolCall } = useToolCallStore();

function connectSession(sessionId) {
    if (ws) {
        ws.close();
    }
    messages.value = [];
    status.value = 'idle';
    streamMap.clear();

    ws = new WebSocket(`ws://127.0.0.1:4010/api/chat/ws/${sessionId}`);

    function routeToolCalls(calls) {
        if (!Array.isArray(calls)) return;

        calls.forEach(call => {
            const fn = call.function;
            const fnName = typeof fn === 'string' ? fn : fn?.name;

            switch (fnName) {
                case 'show_rendered_chart':
                    handleShowRendered(call);
                    break;



                // ✅ 你可以在此添加更多 case：
                case 'show_rendered_element':
                    handleShowRendered(call);
                  break;

                default:
                    console.warn('未处理的工具调用：', fnName, call);
            }
        });
    }

    function handleShowRendered(call) {
        console.log('处理图表渲染调用：', call);
        setSelectedToolCall(call)
        // 可选：你可以在这里发事件、更新状态、调接口等
        // 比如：emit('renderChart', call.function.arguments)
    }
    ws.onmessage = (event) => {
        const data = JSON.parse(event.data);
        // console.log('收到流式消息:', data);
        /*
          假设data结构示例：
          {
            stream_id: 'xxx',
            type: 'content' | 'done' | 'tool_calls' | 'tool_calls_result',
            text: '文本内容',
            info: 工具调用信息,
            role: 'assistant' | 'tool' | ...
            think: true|false,
            ...
          }
        */

        const { stream_id, type, text, info, role, think } = data;

        if (!stream_id) {
            // 非流式消息，或格式异常，忽略或另处理
            return;
        }

        let msg = streamMap.get(stream_id);

        if (!msg) {
            // 新消息，初始化
            msg = {
                stream_id,
                role: role || (type === 'tool_calls_result' ? 'tool' : 'assistant'),
                content: '',
                think: !!think,
            };
            streamMap.set(stream_id, msg);
            messages.value.push(msg);
        }

        if (type === 'content') {
            msg.content += text || '';
        }
        // === 工具调用流式过程（实时 tool_calls_acc 列表） ===
        else if (type === 'tool_call_stream') {
            msg.tool_calls = info || [];
        }
        else if (type === 'tool_calls') {
            msg.tool_calls = info || null;

            if (Array.isArray(info)) {
                // 收集所有 call.id
                info.forEach(call => {
                    if (call.id) {
                        pendingToolCallIds.value.add(call.id);
                    }
                });

                // 调用路由函数进行处理
                routeToolCalls(info);
            }
        } else if (type === 'tool_calls_result') {
            // 工具调用结果直接覆盖内容
            msg.content = info || {};

            pendingToolCallIds.value.delete(info.tool_call_id);
        } else if (type === 'done') {
            // 流结束，更新状态
            status.value = 'completed';
        }

        // 触发响应式更新，刷新界面
        // 先找到 messages 中对应的索引
        const idx = messages.value.findIndex(m => m.stream_id === msg.stream_id);
        if (idx !== -1) {
            messages.value.splice(idx, 1, { ...msg });
        }

        // 保证滚动条滚动到底部（如果需要）
        nextTick(() => {
            const el = document.querySelector('.message-box');
            if (el) {
                el.scrollTop = el.scrollHeight;
            }
        });
    };

    ws.onclose = () => {
        status.value = 'idle';
    };

    ws.onerror = (e) => {
        console.error('[WS] 错误:', e);
    };
}

function sendTask(content) {
    if (ws && content.trim()) {
        messages.value.push({ role: 'user', content });
        ws.send(JSON.stringify({ action: 'start', content }));
        status.value = 'running';
    }
}

function closeWebSocket() {
    if (ws) {
        ws.close();
    }
}

export default function useWebSocket() {
    return {
        messages,
        status,
        connectSession,
        sendTask,
        closeWebSocket,
        pendingToolCallIds
    };
}
