package org.liquibase.core.liquibase;

import liquibase.Liquibase;
import liquibase.changelog.ChangeLogParameters;
import liquibase.changelog.visitor.ChangeExecListener;
import liquibase.changelog.visitor.DefaultChangeExecListener;
import liquibase.database.Database;
import liquibase.exception.DatabaseException;
import liquibase.exception.LiquibaseException;
import liquibase.integration.spring.SpringLiquibase;
import liquibase.integration.spring.SpringResourceAccessor;
import liquibase.resource.ResourceAccessor;
import org.liquibase.core.property.DynamicProperty;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.autoconfigure.liquibase.LiquibaseProperties;
import org.springframework.core.io.DefaultResourceLoader;

import javax.sql.DataSource;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author LENOVO
 * @date 2025/3/14 11:06
 */
public class DynamicSpringLiquibase extends SpringLiquibase implements DisposableBean {

    /**
     * 监听器
     */
    private List<ChangeExecListener> listeners = new ArrayList<>();

    /**
     * 动态属性
     */
    private List<DynamicProperty<?>> dynamicProperties = new ArrayList<>();

    public DynamicSpringLiquibase() {
        super();
    }

    public DynamicSpringLiquibase(LiquibaseProperties properties,
                                  String applicationId,
                                  DataSource dataSource) {
        super();
        setDataSource(dataSource);
        setChangeLog(properties.getChangeLog());
        setClearCheckSums(properties.isClearChecksums());
        setContexts(properties.getContexts());
        setDefaultSchema(properties.getDefaultSchema());
        setLiquibaseSchema(properties.getLiquibaseSchema());
        setLiquibaseTablespace(properties.getLiquibaseTablespace());
        setDatabaseChangeLogTable(properties.getDatabaseChangeLogTable());
        setDatabaseChangeLogLockTable(properties.getDatabaseChangeLogLockTable());
        setDropFirst(properties.isDropFirst());
        setApplicationId(applicationId);
        setShouldRun(properties.isEnabled());
        setLabelFilter(properties.getLabels());
        setChangeLogParameters(properties.getParameters());
        setRollbackFile(properties.getRollbackFile());
        setTestRollbackOnUpdate(properties.isTestRollbackOnUpdate());
        setTag(properties.getTag());
        setResourceLoader(new DefaultResourceLoader());
        setListeners(listeners);
        setDynamicProperties(dynamicProperties);
    }

    @Override
    protected Liquibase createLiquibase(Connection c) throws LiquibaseException {
        Liquibase liquibase = super.createLiquibase(c);
        addListener(liquibase);
        addDynamicProperties(liquibase);
        return liquibase;
    }

    @Override
    protected Database createDatabase(Connection c, ResourceAccessor resourceAccessor) throws DatabaseException {
        return super.createDatabase(c, resourceAccessor);
    }

    @Override
    protected SpringResourceAccessor createResourceOpener() {
        return super.createResourceOpener();
    }

    private void addListener(Liquibase liquibase) {
        DefaultChangeExecListener defaultChangeExecListener = liquibase.getDefaultChangeExecListener();
        listeners.forEach(defaultChangeExecListener::addListener);
        liquibase.setChangeExecListener(defaultChangeExecListener);
    }

//    public void addListeners(List<ChangeExecListener> listeners) {
//        if (listeners != null) {
//            this.listeners.addAll(listeners);
//        }
//    }

    private void addDynamicProperties(Liquibase liquibase) {
        ChangeLogParameters changeLogParameters = liquibase.getChangeLogParameters();
        for (DynamicProperty<?> dynamicProperty : dynamicProperties) {
            Object value = dynamicProperty.getValue();
            if (value == null) {
                continue;
            }
            if (value instanceof Map) {
                Map<String, Object> items = (Map<String, Object>) value;
                items.forEach((key, val) -> {
                    changeLogParameters.set(key, val, null, (String) null,
                            dynamicProperty.getDbms(), true, null);
                });
            } else {
                changeLogParameters.set(dynamicProperty.getName(), dynamicProperty, null, (String) null,
                        dynamicProperty.getDbms(), true, null);
            }
        }
    }

//    public void addDynamicProperties(List<DynamicProperty<?>> dynamicProperties) {
//        if (dynamicProperties != null) {
//            this.dynamicProperties.addAll(dynamicProperties);
//        }
//    }

    @Override
    public void destroy() throws Exception {
        DataSource dataSource = getDataSource();
        if (dataSource instanceof AutoCloseable) {
            AutoCloseable c = (AutoCloseable) dataSource;
            try {
                c.close();
            } catch (Exception ignored) {
            }
        }
    }

    public void setListeners(List<ChangeExecListener> listeners) {
        if (listeners != null) {
            this.listeners.addAll(listeners);
        }
        this.listeners = listeners;
    }

    public void setDynamicProperties(List<DynamicProperty<?>> dynamicProperties) {
        if (dynamicProperties != null) {
            this.dynamicProperties.addAll(dynamicProperties);
        }
    }

    @Override
    public String toString() {
        return getClass().getName() + "(" + beanName + ")";
    }
}
