package com.smart.hospital.common.im.client;

import cn.hutool.core.util.StrUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.smart.hospital.common.im.AbstractImRequest;
import com.smart.hospital.common.im.AbstractImResponse;
import com.smart.hospital.common.im.ImConstants;
import com.smart.hospital.common.im.ImException;
import com.smart.hospital.common.im.util.UserSigUtils;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 抽象IM客户端
 */
public abstract class AbstractImClient implements ImClient {

	private static final String QUESTION_MARK = "?";

	private static final String EQUALS_MARK = "=";

	private static final String AND_MARK = "&";

	/**
	 * objectMapper解析类
	 */
	protected ObjectMapper objectMapper;

	/**
	 * 应用sdkappid
	 */
	private Long sdkAppId;

	/**
	 * 应用key
	 */
	private String key;

	/**
	 * 用户名，调用 REST API 时必须为 App 管理员帐号
	 */
	private String identifier;

	public AbstractImClient(ObjectMapper objectMapper, Long sdkAppId, String key, String identifier) {
		this.objectMapper = objectMapper;
		this.sdkAppId = sdkAppId;
		this.key = key;
		this.identifier = identifier;
	}

	@Override
	public <R extends AbstractImResponse> R getResponse(AbstractImRequest<R> request) {
		Map<String, Object> query = new LinkedHashMap<>(request.query());

		query.put(ImConstants.PARAMS_SDK_APP_ID, sdkAppId);
		query.put(ImConstants.PARAMS_IDENTIFIER, identifier);
		query.put(ImConstants.PARAMS_RANDOM, ThreadLocalRandom.current().nextInt());
		query.put(ImConstants.PARAMS_CONTENT_TYPE, ImConstants.CONTENT_TYPE_JSON);
		try {
			query.put(ImConstants.PARAMS_USERSIG,
					UserSigUtils.genSig(sdkAppId, key, identifier, ImConstants.DEFAULT_EXPIRE));
		} catch (Exception e) {
			throw new ImException(e.getLocalizedMessage(), e);
		}

		StringBuilder queryBuilder = new StringBuilder();

		for (Map.Entry<String, Object> entry : query.entrySet()) {
			String key = entry.getKey();
			Object value = entry.getValue();

			queryBuilder.append(key);
			queryBuilder.append(EQUALS_MARK);
			queryBuilder.append(value);
			queryBuilder.append(AND_MARK);
		}

		StringBuilder uriBuilder = new StringBuilder(ImConstants.BASE_URL);
		uriBuilder.append(request.uri());

		if (uriBuilder.indexOf(QUESTION_MARK) == -1) {
			uriBuilder.append(QUESTION_MARK);
		} else {
			uriBuilder.append(AND_MARK);
		}
		uriBuilder.append(queryBuilder);

		return executeWithHttpRequest(request, StrUtil.removeSuffixIgnoreCase(uriBuilder.toString(), AND_MARK), objectMapper);
	}

	/**
	 * 执行http请求
	 *
	 * @param request      业务请求对象
	 * @param uri          url
	 * @param objectMapper objectMapper
	 * @param <T>          响应类型
	 * @return 响应
	 */
	protected abstract <T extends AbstractImResponse> T executeWithHttpRequest(AbstractImRequest<T> request, String uri, ObjectMapper objectMapper);

}
