package com.smart.hospital.common.lock.advisor.intercept;

import com.smart.hospital.common.lock.advisor.attribute.LockAttribute;
import com.smart.hospital.common.lock.advisor.source.LockAttributeSource;
import com.smart.hospital.common.lock.constant.LockCommonConstant;
import com.smart.hospital.common.lock.key.LockKey;
import com.smart.hospital.common.lock.service.LockService;
import com.smart.hospital.common.lock.service.LockServiceManager;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RedissonClient;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.lang.Nullable;
import org.springframework.util.ConcurrentReferenceHashMap;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;
import java.util.List;
import java.util.concurrent.ConcurrentMap;

@Slf4j
public abstract class LockInterceptorSupport implements InitializingBean, ApplicationContextAware {

	@Nullable
	private ApplicationContext applicationContext;

	private LockAttributeSource lockAttributeSource;

	private RedissonClient redissonClient;

	@Autowired
	private LockServiceManager lockServiceManager;

	@Nullable
	private String redissonClientBeanName = LockCommonConstant.LOCK_CLIENT_BEAN_NAME;

	private final ConcurrentMap<Object, RedissonClient> clientCache =
			new ConcurrentReferenceHashMap<>(4);

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		if (this.applicationContext == null && getRedissonClient() == null) {
			throw new IllegalArgumentException(
					"Set the 'redissonClient' property or make sure to run within a BeanFactory " +
							"containing a RedissonClient bean!");
		}
		if (getLockAttributeSource() == null) {
			throw new IllegalArgumentException("LockAttributeSource' is required:" + "没用配置资源解析类");
		}
	}

	private RedissonClient determineRedissonClient() {
		RedissonClient redissonClient = clientCache.get(this.redissonClientBeanName);
		if (redissonClient != null) {
			return redissonClient;
		}
		if (this.applicationContext == null && getRedissonClient() != null) {
			return getRedissonClient();
		}
		RedissonClient bean;
		if (StringUtils.hasText(this.redissonClientBeanName)) {
			bean = (RedissonClient) applicationContext.getBean(redissonClientBeanName);
		} else {
			bean = applicationContext.getBean(RedissonClient.class);
		}

		if (bean == null) {
			throw new NoSuchBeanDefinitionException("No matching bean found for bean name : " + this.redissonClientBeanName);
		}
		this.redissonClient = bean;
		clientCache.put(this.redissonClientBeanName, bean);
		return bean;
	}

	protected Object invocationWithLock(Method method, Class<?> targetClass, Object[] arguments, InvocationCallback invocation) throws Throwable {
		Object retVal;
		LockKey<String, List<String>> lockKey;
		LockService lockService = null;
		try {
			LockAttributeSource as = getLockAttributeSource();
			LockAttribute attr = as.getLockAttribute(method, targetClass);
			if (attr != null) {
				// 在这里解析@Key注解的信息
				lockKey = as.getLockKey(method, targetClass, arguments, attr);
				if (lockKey != null) {
					lockService = lockServiceManager.findLockService(lockKey);
					if (lockService != null) {
						lockService.lock();
					}
				}
			}
			retVal = invocation.proceedWithInvocation();
		} finally {
			if (lockService != null) {
				lockService.release();
			}
		}
		return retVal;
	}


	@FunctionalInterface
	protected interface InvocationCallback {
		Object proceedWithInvocation() throws Throwable;
	}

	public void setRedissonClientBeanName(@Nullable String redissonClientBeanName) {
		this.redissonClientBeanName = redissonClientBeanName;
	}

	public LockAttributeSource getLockAttributeSource() {
		return lockAttributeSource;
	}

	public void setLockAttributeSource(LockAttributeSource lockAttributeSource) {
		this.lockAttributeSource = lockAttributeSource;
	}

	public void setRedissonClient(RedissonClient redissonClient) {
		this.redissonClient = redissonClient;
	}

	public RedissonClient getRedissonClient() {
		return this.redissonClient;
	}
}
