package com.smart.hospital.common.lock.key.strategy;

import com.smart.hospital.common.lock.advisor.attribute.LockAttribute;
import com.smart.hospital.common.lock.constant.LockCommonConstant;
import com.smart.hospital.common.lock.enums.LockScheme;
import com.smart.hospital.common.lock.key.LockKey;
import com.smart.hospital.common.lock.key.RedisLockKey;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

public abstract class AbstractKeyStrategy implements KeyStrategy {

	/**
	 * 锁类型
	 */
	protected LockScheme scheme;

	/**
	 * 类名称
	 */
	protected String className;

	/**
	 * 方法名称
	 */
	protected String methodName;

	/**
	 * 方法
	 */
	protected Method realMethod;

	/**
	 * 方法参数
	 */
	protected Object[] args;

	/**
	 * @Lock的属性值
	 */
	protected LockAttribute attribute;

	public AbstractKeyStrategy(String className, String methodName, Method realMethod, Object[] args, LockScheme scheme, LockAttribute attribute) {
		this.className = className;
		this.methodName = methodName;
		this.realMethod = realMethod;
		this.args = args;
		this.scheme = scheme;
		this.attribute = attribute;
	}

	/**
	 * 获取类名称
	 *
	 * @return
	 */
	public String getSimpleClassName() {
		String[] split = this.className.split("\\.");
		return split[split.length - 1];
	}

	/**
	 * 包装锁的key(在redis里的key)
	 *
	 * @param valTag
	 * @return
	 */
	protected String wrapKey(String valTag) {
		return getSimpleClassName() + LockCommonConstant.KEY_SPLIT_MARK + this.methodName + LockCommonConstant.KEY_SPLIT_MARK + valTag;
	}

	protected LockKey<String, List<String>> buildLockKey() {
		LockKey<String, List<String>> lockKey;
		if (this.scheme == LockScheme.REDIS) {
			lockKey = new RedisLockKey(this.attribute.getLockType(), this.scheme, attribute.getLeaseTime(), attribute.getWaitTime(), attribute.getTimeUnit(), new ArrayList<>());
		} else {
			throw new IllegalArgumentException("目前只支持Redis");
		}
		return lockKey;
	}

	@Override
	public LockKey<String, List<String>> buildKey() {
		LockKey<String, List<String>> key = buildLockKey();
		buildKeyItem(key);
		return key;
	}

	protected abstract void buildKeyItem(LockKey<String, List<String>> lockKey);

	public LockScheme getScheme() {
		return scheme;
	}

	public void setScheme(LockScheme scheme) {
		this.scheme = scheme;
	}

	public String getClassName() {
		return className;
	}

	public void setClassName(String className) {
		this.className = className;
	}

	public String getMethodName() {
		return methodName;
	}

	public void setMethodName(String methodName) {
		this.methodName = methodName;
	}

	public Method getRealMethod() {
		return realMethod;
	}

	public void setRealMethod(Method realMethod) {
		this.realMethod = realMethod;
	}

	public Object[] getArgs() {
		return args;
	}

	public void setArgs(Object[] args) {
		this.args = args;
	}

	public LockAttribute getAttribute() {
		return attribute;
	}

	public void setAttribute(LockAttribute attribute) {
		this.attribute = attribute;
	}
}
