package com.smart.hospital.common.mq.factory;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.smart.hospital.common.mq.annotation.Id;
import com.smart.hospital.common.mq.annotation.Exchange;
import com.smart.hospital.common.mq.annotation.Message;
import com.smart.hospital.common.mq.annotation.RoutingKey;
import com.smart.hospital.common.mq.enums.DeliverStatus;
import com.smart.hospital.common.mq.handler.MessageContentHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.connection.CorrelationData;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.lang.UsesJava7;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.*;

@Slf4j
public class ClientProxy<T> implements InvocationHandler, Serializable {

	private static final long serialVersionUID = 1L;

	private final RabbitTemplate rabbitTemplate;

	private final Class<T> mapperInterface;

	private final MessageContentHandler messageContentHandler;

	public ClientProxy(RabbitTemplate rabbitTemplate, Class<T> mapperInterface, MessageContentHandler messageContentHandler) {
		this.rabbitTemplate = rabbitTemplate;
		this.mapperInterface = mapperInterface;
		this.messageContentHandler = messageContentHandler;
	}

	@Override
	public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
		if (Object.class.equals(method.getDeclaringClass())) {
			return method.invoke(this, args);
		} else if (isDefaultMethod(method)) {
			return invokeDefaultMethod(proxy, method, args);
		}
		Parameter[] parameters = method.getParameters();

		String exchange = "";
		String routingKey = "";
		Object message = null;
		String id = null;
		for (int i = 0; i < parameters.length; i++) {
			Parameter parameter = parameters[i];
			Object arg = args[i];
			if (ObjectUtil.isNull(arg)) {
				continue;
			}
			if (parameter.isAnnotationPresent(Exchange.class)) {
				exchange = String.valueOf(arg);
			} else if (parameter.isAnnotationPresent(RoutingKey.class)) {
				routingKey = String.valueOf(arg);
			} else if (parameter.isAnnotationPresent(Message.class)) {
				message = arg;
			} else if (parameter.isAnnotationPresent(Id.class)) {
				id = String.valueOf(arg);
			}
		}
		if (message == null) {
			throw new IllegalArgumentException("消息队列的消息不能为空!");
		}
		if (StrUtil.isBlank(id)) {
			id = IdUtil.fastUUID();
		}
		// 进行消息存储
		if (messageContentHandler != null) {
			messageContentHandler.addMessage(exchange, routingKey, message, id, DeliverStatus.DELIVERING.getCode());
		}
		rabbitTemplate.convertAndSend(exchange, routingKey, message, new CorrelationData(id));
		return null;
	}

	@UsesJava7
	private Object invokeDefaultMethod(Object proxy, Method method, Object[] args)
			throws Throwable {
		final Constructor<MethodHandles.Lookup> constructor = MethodHandles.Lookup.class
				.getDeclaredConstructor(Class.class, int.class);
		if (!constructor.isAccessible()) {
			constructor.setAccessible(true);
		}
		final Class<?> declaringClass = method.getDeclaringClass();
		return constructor
				.newInstance(declaringClass,
						MethodHandles.Lookup.PRIVATE | MethodHandles.Lookup.PROTECTED
								| MethodHandles.Lookup.PACKAGE | MethodHandles.Lookup.PUBLIC)
				.unreflectSpecial(method, declaringClass).bindTo(proxy).invokeWithArguments(args);
	}

	/**
	 * Backport of java.lang.reflect.Method#isDefault()
	 */
	private boolean isDefaultMethod(Method method) {
		return (method.getModifiers()
				& (Modifier.ABSTRACT | Modifier.PUBLIC | Modifier.STATIC)) == Modifier.PUBLIC
				&& method.getDeclaringClass().isInterface();
	}

	/**
	 * 获取接口上的泛型T
	 *
	 * @param o     接口
	 * @param index 泛型索引
	 */
	public Class<?> getInterfaceT(Object o, int index) {
		Type[] types = o.getClass().getGenericInterfaces();
		ParameterizedType parameterizedType = (ParameterizedType) types[0];
		Type type = parameterizedType.getActualTypeArguments()[index];
		return checkType(type, index);
	}

	/**
	 * 获取类上的泛型T
	 *
	 * @param o     接口
	 * @param index 泛型索引
	 */
	public static Class<?> getClassT(Object o, int index) {
		Type type = o.getClass().getGenericSuperclass();
		if (type instanceof ParameterizedType) {
			ParameterizedType parameterizedType = (ParameterizedType) type;
			Type actType = parameterizedType.getActualTypeArguments()[index];
			return checkType(actType, index);
		} else {
			String className = type == null ? "null" : type.getClass().getName();
			throw new IllegalArgumentException("Expected a Class, ParameterizedType"
					+ ", but <" + type + "> is of type " + className);
		}
	}

	private static Class<?> checkType(Type type, int index) {
		if (type instanceof Class<?>) {
			return (Class<?>) type;
		} else if (type instanceof ParameterizedType) {
			ParameterizedType pt = (ParameterizedType) type;
			Type t = pt.getActualTypeArguments()[index];
			return checkType(t, index);
		} else {
			String className = type == null ? "null" : type.getClass().getName();
			throw new IllegalArgumentException("Expected a Class, ParameterizedType"
					+ ", but <" + type + "> is of type " + className);
		}
	}
}
