package com.smart.hospital.common.mq.factory;

import com.smart.hospital.common.mq.handler.MessageContentHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.ClassPathBeanDefinitionScanner;

import java.util.Arrays;
import java.util.Set;

/**
 * RabbitClient资源扫描器
 */
@Slf4j
public class RabbitClientScanner extends ClassPathBeanDefinitionScanner {

	private ClientFactoryBean<?> clientFactoryBean = new ClientFactoryBean<>();

	private ApplicationContext applicationContext;

	private RabbitTemplate rabbitTemplate;

	private String rabbitTemplateBeanName;

	private MessageContentHandler messageContentHandler;

	public RabbitClientScanner(BeanDefinitionRegistry registry) {
		super(registry, false);
	}

	@Override
	protected Set<BeanDefinitionHolder> doScan(String... basePackages) {
		Set<BeanDefinitionHolder> beanDefinitions = super.doScan(basePackages);
		if (beanDefinitions.isEmpty()) {
			log.warn("No RabbitClient was found in '" + Arrays.toString(basePackages) + "' package. Please check your configuration.");
		} else {
			processBeanDefinitions(beanDefinitions);
		}
		return beanDefinitions;
	}

	@Override
	protected boolean isCandidateComponent(AnnotatedBeanDefinition beanDefinition) {
		return beanDefinition.getMetadata().isInterface() && beanDefinition.getMetadata().isIndependent();
	}

	private void processBeanDefinitions(Set<BeanDefinitionHolder> beanDefinitions) {
		GenericBeanDefinition definition;
		for (BeanDefinitionHolder holder : beanDefinitions) {
			definition = (GenericBeanDefinition) holder.getBeanDefinition();
			log.debug("Creating ClientFactoryBean with name '" + holder.getBeanName() + "' and '" + definition.getBeanClassName() + "' clientInterface");
			// 构造方法注入
			definition.getConstructorArgumentValues().addGenericArgumentValue(definition.getBeanClassName());
			// 属性注入
			if (this.rabbitTemplate != null) {
				definition.getPropertyValues().add("rabbitTemplate", this.rabbitTemplate);
			}
			if (this.rabbitTemplateBeanName != null) {
				definition.getPropertyValues().add("rabbitTemplate", new RuntimeBeanReference(this.rabbitTemplateBeanName));
			}
			definition.getPropertyValues().add("applicationContext", this.applicationContext);
			definition.getPropertyValues().add("messageContentHandler", this.messageContentHandler);
			// 设置FactoryBean类型
			definition.setBeanClass(clientFactoryBean.getClass());
			definition.setPrimary(true);
			definition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE);
		}
	}

	public void registerFilters() {

		boolean acceptAllInterfaces = true;

		if (acceptAllInterfaces) {
			// default include filter that accepts all classes
			addIncludeFilter((metadataReader, metadataReaderFactory) -> true);
		}

		// exclude package-info.java
		addExcludeFilter((metadataReader, metadataReaderFactory) -> {
			String className = metadataReader.getClassMetadata().getClassName();
			return className.endsWith("package-info");
		});
	}

	public void setClientFactoryBean(ClientFactoryBean<?> clientFactoryBean) {
		this.clientFactoryBean = clientFactoryBean;
	}

	public void setRabbitTemplate(RabbitTemplate rabbitTemplate) {
		this.rabbitTemplate = rabbitTemplate;
	}

	public void setRabbitTemplateBeanName(String rabbitTemplateBeanName) {
		this.rabbitTemplateBeanName = rabbitTemplateBeanName;
	}

	public void setApplicationContext(ApplicationContext applicationContext) {
		this.applicationContext = applicationContext;
	}

	public void setMessageContentHandler(MessageContentHandler messageContentHandler) {
		this.messageContentHandler = messageContentHandler;
	}

}
