package com.smart.hospital.common.pay.source;

import org.springframework.core.MethodClassKey;
import org.springframework.lang.Nullable;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public abstract class AbstractPayAttributeSource implements PayAttributeSource {

	private final Map<Object, PayAttribute> attributeCache = new ConcurrentHashMap<>(16);

	@Override
	public PayAttribute getPayAttribute(Method method, Class<?> targetClass) {
		if (method.getDeclaringClass() == Object.class) {
			return null;
		}

		// 从本地缓存中读取
		Object cacheKey = getCacheKey(method, targetClass);
		PayAttribute payAttribute = this.attributeCache.get(cacheKey);
		if (payAttribute != null) {
			return payAttribute;
		}

		// 解析注解信息
		PayAttribute attr = computePayAttribute(method, targetClass);
		if (attr != null) {
			//  将信息放入本地缓存
			this.attributeCache.put(cacheKey, attr);
		}

		return attr;
	}

	protected PayAttribute computePayAttribute(Method method, Class<?> targetClass) {
		PayAttribute payAttribute = findPayAttribute(method);
		if (payAttribute != null) {
			return payAttribute;
		}
		return null;
	}

	protected abstract PayAttribute findPayAttribute(Method method);

	protected Object getCacheKey(Method method, @Nullable Class<?> targetClass) {
		return new MethodClassKey(method, targetClass);
	}
}
