const mysql = require('mysql2/promise');
const readline = require('readline');
const fs = require('fs');

const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout,
});

// Function to prompt user for input
function askQuestion(query) {
  return new Promise((resolve) => {
    rl.question(query, resolve);
  });
}

async function main() {
  try {
    // Get database connection details from user
    const host = 'localhost';
    const port = '3306';
    const user = 'root';
    const password = '12345678';
    const database = 'teaching';

    // const host = '10.10.10.132';
    // const port = '3306';
    // const user = 'root';
    // const password = 'fxkc@2024';
    // const database = 'data_governance_mining';

    let tableNames = await askQuestion(
      'Enter table names (separated by commas): '
    );
    if (!tableNames) {
      console.log('表名不得为空！');
      return;
    }

    // Split table names by both Chinese and English commas, and trim whitespace
    tableNames = tableNames
      .split(/[,，]/)
      .map((name) => name.trim())
      .filter((name) => name);

    // Process each table name
    for (const tableName of tableNames) {
      // Check if table name contains Chinese characters
      if (/[\u4e00-\u9fa5]/.test(tableName)) {
        console.error(`Table name "${tableName}" 不能有中文.`);
        continue;
      }

      // Convert camelCase to snake_case
      let processedTableName = tableName
        .replace(/([A-Z])/g, '_$1')
        .toLowerCase();
      // Remove leading underscore if present
      if (processedTableName[0] === '_') {
        processedTableName = processedTableName.slice(1);
      }
      console.log('Processing table:', processedTableName);

      console.log('\nConnecting to database...');

      // Create connection
      const connection = await mysql.createConnection({
        host,
        port: parseInt(port),
        user,
        password,
        database,
      });

      console.log('Connected to database successfully!\n');

      // Execute query
      const sql = `SELECT COLUMN_NAME, COLUMN_COMMENT FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ? order by ordinal_position`;

      console.log('Executing query...\n');
      const [rows] = await connection.execute(sql, [
        database,
        processedTableName,
      ]);

      if (rows.length === 0) {
        console.error(
          `***** No results found for table ${processedTableName}. *****`
        );
      } else {
        generateXMLWithTableColumns(processedTableName, rows);
        console.log(
          `===== XML file generated successfully for ${processedTableName}! =====`
        );
      }

      // Close connection
      await connection.end();
      console.log('\nConnection closed.');
    }
  } catch (error) {
    console.error('Error:', error.message);
  } finally {
    rl.close();
  }
}

function generateXMLWithTableColumns(tableName, rows) {
  // XML template start with the table name in the label
  let xmlContent = `<mxGraphModel>
    <root>
        <mxCell id="0"/>
        <mxCell id="1" parent="0"/>
        <object label="${tableName}" id="2">
            <mxCell style="swimlane;fontStyle=1;childLayout=stackLayout;horizontal=1;startSize=30;horizontalStack=0;resizeParent=1;resizeParentMax=0;resizeLast=0;collapsible=1;marginBottom=0;fillColor=#dae8fc;strokeColor=#6c8ebf;" vertex="1" parent="1">
                <mxGeometry x="340" y="140" width="260" height="${
                  30 + rows.length * 30
                }" as="geometry"/>
            </mxCell>
        </object>`;

  // Generate a cell for each column
  rows.forEach((row, index) => {
    const cellValue = `${row.COLUMN_NAME}    ${row.COLUMN_COMMENT || ''}`;
    const cellId = index + 3; // Starting from 3

    // Check if the column is an audit field
    const unimportantColumns = [
      'created_at',
      'create_time',
      'created_code',
      'create_code',
      'updated_at',
      'update_time',
      'updated_code',
      'update_code',
      'deleted_at',
      'deleted_code',
      'delete_flag',
    ].includes(row.COLUMN_NAME);
    const style = unimportantColumns
      ? 'text;strokeColor=default;fillColor=none;align=left;verticalAlign=middle;spacingLeft=4;spacingRight=4;overflow=hidden;points=[[0,0.5],[1,0.5]];portConstraint=eastwest;rotatable=0;fontColor=#B3B3B3;'
      : 'text;strokeColor=default;fillColor=none;align=left;verticalAlign=middle;spacingLeft=4;spacingRight=4;overflow=hidden;points=[[0,0.5],[1,0.5]];portConstraint=eastwest;rotatable=0;';

    xmlContent += `
        <mxCell id="${cellId}" value="${cellValue}" style="${style}" vertex="1" parent="2">
            <mxGeometry y="${
              30 + index * 30
            }" width="260" height="30" as="geometry"/>
        </mxCell>`;
  });

  // XML template end
  xmlContent += `
    </root>
</mxGraphModel>`;
  // console.log(xmlContent);
  fs.writeFileSync(`xml/${tableName}_diagram.xml`, xmlContent);
}

main();
