import getConfig from 'next/config';
import { backendApi } from '.';
import { getModel } from '@/utils/gpt';
import { fetchEventSource, EventStreamContentType } from '@microsoft/fetch-event-source';
import { MutableRefObject, SetStateAction } from 'react';
import { functions, get } from 'lodash';

export interface View {
    type: 'schema' | 'table';
    name?: string;
}

const model = getModel(process.env.NEXT_PUBLIC_MODEL_NAME || 'gpt-4o-mini');

export function prettyObject(msg: any) {
    const obj = msg;
    if (typeof msg !== 'string') {
        msg = JSON.stringify(msg, null, '  ');
    }
    if (msg === '{}') {
        return obj.toString();
    }
    if (msg.startsWith('```json')) {
        return msg;
    }
    return ['```json', msg, '```'].join('\n');
}

type message = {
    role: string;
    content: string;
};

export default class OpenAI {
    static async request(
        messages: message[],
        onFinish: (responseText: string, cancel?: boolean) => any,
        onUpdate?: (responseText: string, delta: string) => any,
        onError?: (e: Error) => any,
        functions?: any[],
        closeFn?: MutableRefObject<any>,
        function_call?: any,
        stream: boolean = true
    ) {
        const requestPayload = {
            messages: messages,
            model: model.name,
            temperature: model.temperature,
            frequency_penalty: model.frequency_penalty,
            presence_penalty: model.presence_penalty,
            stream,
            functions,
            function_call,
        };
        const controller = new AbortController();
        const chatPayload = {
            method: 'POST',
            body: JSON.stringify(requestPayload),
            signal: controller.signal,
            headers: {
                'cache-control': 'no-cache',
                'Content-Type': 'application/json',
                'x-requested-with': 'XMLHttpRequest',
                Accept: 'text/event-stream',
            },
        };
        const requestTimeoutId = setTimeout(() => controller.abort(), 1000 * 120);

        const chatPath = '/openai/v1/chat/completions';
        if (closeFn) {
            controller.signal.onabort = () => {
                onFinish('', true);
            };
            closeFn.current = () => {
                clearTimeout(requestTimeoutId);
                controller.abort();
            };
        }
        if (stream) {
            let responseText = '';
            let finished = false;
            const finish = (cancel?: boolean) => {
                if (!finished) {
                    onFinish(responseText, cancel);
                    finished = true;
                }
            };
            controller.signal.onabort = () => {
                finish(true);
            };
            fetchEventSource(chatPath, {
                ...chatPayload,
                async onopen(res) {
                    clearTimeout(requestTimeoutId);
                    const contentType = res.headers.get('content-type');
                    // console.log('[OpenAI] request response content type: ', contentType);

                    if (contentType?.startsWith('text/plain')) {
                        responseText = await res.clone().text();
                        return finish();
                    }

                    if (
                        !res.ok ||
                        !res.headers.get('content-type')?.startsWith(EventStreamContentType) ||
                        res.status !== 200
                    ) {
                        const responseTexts = [responseText];
                        let extraInfo = await res.clone().text();
                        try {
                            const resJson = await res.clone().json();
                            extraInfo = prettyObject(resJson);
                        } catch {}

                        if (extraInfo) {
                            responseTexts.push(extraInfo);
                        }

                        responseText = responseTexts.join('\n\n');

                        return finish();
                    }
                },
                onmessage(msg) {
                    if (msg.data === '[DONE]' || finished) {
                        return finish();
                    }
                    const text = msg.data;
                    try {
                        // console.log(text);
                        const json = JSON.parse(text);
                        const delta = json.choices[0].delta.content;
                        if (json.choices[0].finish_reason === 'function_call') {
                            // 🌟 output the Tree structure data
                            console.log(json);
                        }
                        if (delta) {
                            responseText += delta;
                            onUpdate?.(responseText, delta);
                        }
                    } catch (e) {
                        console.error('[Request] parse error', text, msg);
                    }
                },
                onclose() {
                    finish();
                },
                onerror(e) {
                    onError?.(e);
                    throw e;
                },
                openWhenHidden: true,
            });
        } else {
            const res = await fetch(chatPath, chatPayload);
            clearTimeout(requestTimeoutId);
            const resJson = await res.json();
            const message =
                resJson.choices?.at(0)?.message?.content ??
                (JSON.stringify(get(resJson, 'choices[0].message.function_call')) || '');
            onFinish(message);
        }
    }
}
