'use client';
import React, {
  useEffect,
  useRef,
  useCallback,
  useReducer,
  type Provider as ReactProvider,
  useMemo,
  useState,
} from 'react';
import { Graph, Path, Cell } from '@antv/x6';
import { Selection } from '@antv/x6-plugin-selection';
// import insertCss from 'insert-css';
import { History } from '@antv/x6-plugin-history';
import { DagreLayout } from '@antv/layout';
import styled from 'styled-components';
import { Button, Collapse, Layout, List } from '@arco-design/web-react';
import { IconLayout } from '@arco-design/web-react/icon';

import { Dnd } from '@antv/x6-plugin-dnd';
import { MiniMap } from '@antv/x6-plugin-minimap';
import nodeStatusList from './nodeStatus';
import { NodeStatus, NodeType } from './nodes/AlgoNode';
import ConnectorInit from './connectors';
import EdgesInit from './edges';
import nodesInit from './nodes';
import { Div } from './styleWidgets';
import { SlideForm } from './nodeConfigs';
import getDagContext, { DagActions } from './context';
import { produce } from 'immer';
import { set, uniqueId } from 'lodash';
import { Portal } from '@antv/x6-react-shape';

const CollapseItem = Collapse.Item;

//注册
ConnectorInit();
EdgesInit();
nodesInit();

const Sider = Layout.Sider;
const Header = Layout.Header;
const Content = Layout.Content;
const X6ReactPortalProvider = Portal.getProvider();

function A() {
  console.log('A');
  return <div>A</div>;
}

function useFirst(fn: typeof getDagContext) {
  const flag = useRef<ReturnType<typeof fn>>();
  if (!flag.current) {
    flag.current = fn();
  }
  return flag.current;
}

export default function Demo() {
  const el = useRef<HTMLDivElement>(null);
  const graphEl = useRef<Graph | null>();
  const dndEl = useRef<Dnd | null>();
  const context = useFirst(getDagContext);
  const graph = graphEl.current;
  const dnd = dndEl.current;
  const { Provider, DagContext } = context;
  console.log(DagContext, 'DagContext');

  const [state, dispatch] = useReducer(
    produce(
      (
        state: typeof Provider extends ReactProvider<infer P> ? P : never,
        action: {
          type: DagActions;
          payload: (typeof action)['type'] extends DagActions.CHANGE_ACTIVE_CELL ? Cell : never;
        }
      ) => {
        switch (action.type) {
          case DagActions.CHANGE_ACTIVE_CELL: {
            console.info(action);
            state.activeCell = action.payload;
          }
        }
      }
    ),
    {}
  );

  const store = useMemo(() => {
    return {
      ...state,
      dispatch: dispatch,
    };
  }, [state]);
  console.log(store);

  useEffect(() => {
    if (el) {
      const w = (el?.current?.clientWidth || 400) - 400;
      const h = el?.current?.clientHeight;

      const graph: Graph & {
        DagContext?: typeof DagContext;
      } = (graphEl.current = new Graph({
        container: el.current!,
        width: w,
        height: 418,
        panning: {
          enabled: true,
          eventTypes: ['leftMouseDown', 'mouseWheel'],
        },
        mousewheel: {
          enabled: true,
          modifiers: 'ctrl',
          factor: 1.1,
          maxScale: 1.5,
          minScale: 0.5,
        },
        highlighting: {
          magnetAdsorbed: {
            name: 'stroke',
            args: {
              attrs: {
                fill: '#fff',
                stroke: '#31d0c6',
                strokeWidth: 4,
              },
            },
          },
        },
        connecting: {
          snap: true,
          allowBlank: false,
          allowLoop: false,
          highlight: true,
          connector: 'algo-connector',
          connectionPoint: 'anchor',
          anchor: 'center',
          validateMagnet({ magnet }) {
            return magnet.getAttribute('port-group') !== 'top';
          },
          createEdge() {
            return graph.createEdge({
              shape: 'dag-edge',
              attrs: {
                line: {
                  strokeDasharray: '5 5',
                },
              },
              zIndex: -1,
            });
          },
        },
      } as any));

      graph.DagContext = DagContext;

      const dnd = (dndEl.current = new Dnd({
        target: graph,
      }));
      // graph.use(
      //   new MiniMap({
      //     container: document.getElementById('minimap'),
      //   }),
      // )

      graph.use(
        new Selection({
          multiple: true,
          rubberEdge: true,
          rubberNode: true,
          modifiers: 'shift',
          rubberband: true,
        })
      );

      graph.use(
        new History({
          enabled: true,
        })
      );

      graph.on('edge:connected', ({ edge }) => {
        edge.attr({
          line: {
            strokeDasharray: '',
          },
        });
      });

      graph.on('node:change:data', ({ node }) => {
        const edges = graph.getIncomingEdges(node);
        const { status } = node.getData() as NodeStatus;
        edges?.forEach(edge => {
          if (status === 'running') {
            edge.attr('line/strokeDasharray', 5);
            edge.attr('line/style/animation', 'running-line 30s infinite linear');
          } else {
            edge.attr('line/strokeDasharray', '');
            edge.attr('line/style/animation', '');
          }
        });
      });

      graph.on('node:contextmenu', ({ node, view }) => {
        dispatch({
          type: DagActions.CHANGE_ACTIVE_CELL,
          payload: node,
        });
      });
      // 初始化节点/边
      const init = (data: Cell.Metadata[]) => {
        const cells: Cell[] = [];
        data.forEach(item => {
          cells.push(graph.createNode(item));
        });
        graph.resetCells(cells);
      };

      init([
        {
          id: 'flow' + '_' + Date.now(),
          shape: 'dag-node',
          x: 290,
          y: 110,
          data: {
            label: '流程开始',
            status: 'success',
            nodeType: NodeType.FLOW_START,
          },
          ports: [],
        },
      ]);

      graph.centerContent();
      return () => {
        graph.dispose();
      };
    }
  }, []);

  const startDrag = (e: React.MouseEvent<HTMLDivElement, MouseEvent>) => {
    // 该 node 为拖拽的节点，默认也是放置到画布上的节点，可以自定义任何属性
    if (graph && dnd) {
      const node = graph.createNode({
        shape: 'rect',
        width: 100,
        height: 40,
      });
      dnd.start(node, e.nativeEvent);
    }
  };

  return (
    <Div className="react-portal-app">
      <Layout className="h-[500px]">
        <Header className="h-[40px] flex pl-[200px] bg-slate-50 border-b border-b-gray-200 relative z-[1] bg-clip-content">
          <div className="px-[10px]">
            <Button
              iconOnly
              icon={<IconLayout />}
              onClick={() => {
                // const dagreLayout = new DagreLayout({
                //   type: 'dagre',
                //   rankdir: 'LR',
                //   align: 'UR',
                //   ranksep: 35,
                //   nodesep: 15,
                // });
                // const model = dagreLayout.layout({
                //   nodes: graphEl.current?.getNodes().map((v,i) => {
                //     return {
                //       id: `${v.id}`,
                //       shape: 'circle',
                //       width: 32,
                //       height: 32,
                //       label: i,
                //       attrs: {
                //         body: {
                //           fill: '#5F95FF',
                //           stroke: 'transparent',
                //         },
                //         label: {
                //           fill: '#ffffff',
                //         },
                //       },
                //     };
                //   }),
                //   edges: graphEl.current?.getEdges(),
                // });
                // console.log(model);
                // graphEl.current?.fromJSON(model);
              }}
            />
          </div>
        </Header>
        <Layout>
          <Sider className="bg-slate-100 mt-[-40px] border-r border-r-gray-200 pt-[40px]">
            <Collapse bordered={false} defaultActiveKey={['1']}>
              <CollapseItem header="Query" name="1">
                <List
                  style={{ width: '100%' }}
                  size="small"
                  header={null}
                  dataSource={['查询1', '查询2']}
                  render={(item, index) => (
                    <div className="m-[10px] text-center bg-slate-300 p-[5px] rounded cursor-pointer text-purple-400 font-bold shadow">
                      <span className="w-full" key={index}>
                        {item}
                      </span>
                    </div>
                  )}
                />
              </CollapseItem>
            </Collapse>
          </Sider>
          <Content>
            <Provider value={store}>
              <X6ReactPortalProvider />
            </Provider>
            <div ref={el}></div>
          </Content>
          <Sider
            className="bg-slate-100  mt-[-40px] border-l border-l-gray-200  pt-[40px]"
            collapsible
            collapsedWidth={0}
          >
            <SlideForm />
          </Sider>
        </Layout>
      </Layout>
    </Div>
  );
}
