import { NextRequest, NextResponse } from 'next/server';

const url = new URL(`${process.env.OPENAI_PROXY_URL || 'https://api.openai.com'}`);
const PROTOCOL = url.protocol;
const BASE_URL = url.host;
const DISABLE_GPT4 = !!process.env.DISABLE_GPT4;
console.log(PROTOCOL, BASE_URL);

export async function requestOpenai(req: NextRequest) {
    const controller = new AbortController();
    const authValue = `Bearer ${process.env.NEXT_PUBLIC_OPEN_AI_API_KEY || ''}`;
    const openaiPath = `${req.nextUrl.pathname}${req.nextUrl.search}`.replaceAll('/openai/', '');

    let baseUrl = BASE_URL;

    if (!baseUrl.startsWith('http')) {
        baseUrl = `${PROTOCOL}//${baseUrl}`;
    }

    console.log('[Proxy] ', openaiPath);
    console.log('[Base Url]', baseUrl);

    if (process.env.OPENAI_ORG_ID) {
        console.log('[Org ID]', process.env.OPENAI_ORG_ID);
    }

    const timeoutId = setTimeout(() => {
        controller.abort();
    }, 10 * 60 * 1000);

    const fetchUrl = `${baseUrl}/${openaiPath}`;
    const fetchOptions: RequestInit = {
        headers: {
            'Content-Type': 'application/json',
            Authorization: authValue,
            ...(process.env.OPENAI_ORG_ID && {
                'OpenAI-Organization': process.env.OPENAI_ORG_ID,
            }),
        },
        cache: 'no-store',
        method: req.method,
        body: req.body,
        // @ts-ignore
        duplex: 'half',
        signal: controller.signal,
    };

    // #1815 try to refuse gpt4 request
    if (DISABLE_GPT4 && req.body) {
        try {
            const clonedBody = await req.text();
            fetchOptions.body = clonedBody;

            const jsonBody = JSON.parse(clonedBody);

            if ((jsonBody?.model ?? '').includes('gpt-4')) {
                return NextResponse.json(
                    {
                        error: true,
                        message: 'you are not allowed to use gpt-4 model',
                    },
                    {
                        status: 403,
                    }
                );
            }
        } catch (e) {
            console.error('[OpenAI] gpt4 filter', e);
        }
    }

    try {
        const res = await fetch(fetchUrl, fetchOptions);

        // to prevent browser prompt for credentials
        const newHeaders = new Headers(res.headers);
        newHeaders.delete('www-authenticate');

        // to disbale ngnix buffering
        newHeaders.set('X-Accel-Buffering', 'no');

        return new Response(res.body, {
            status: res.status,
            statusText: res.statusText,
            headers: res.headers,
        });
    } finally {
        clearTimeout(timeoutId);
    }
}

export async function middleware(request: NextRequest) {
    // Only process requests that start with /openai/
    console.log(new URL(request.url).pathname, 'targetUrl');
    const path = new URL(request.url).pathname;
    if (!path.startsWith('/openai/')) {
        return NextResponse.next();
    }
    try {
        return await requestOpenai(request);
    } catch (e) {
        console.log(e);
    }
}
