import { Upload, Message, UploadProps } from '@arco-design/web-react';
import * as XLSX from 'xlsx';
import { useState } from 'react';
import pinyin from 'pinyin';
import { flatten, get, isDate, isString, map } from 'lodash';

const e = /^\s+$/g;
function getDataType(value: string) {
    if (!value || (isString(value) && e.test(value))) {
        console.log(value, 'value');
        return 'VARCHAR(255)';
    } else if (isDate(value) && !isNaN(Date.parse(value))) {
        if (/^\d{4}-\d{2}-\d{2}$/.test(value)) {
            return 'DATE';
        } else {
            if (/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}(\.\d{1,3})?$/.test(value)) {
                return 'DATETIME';
            }
            return 'VARCHAR(255)';
        }
    } else if (isFinite(Number(value))) {
        const number = Number(value);
        if (Number.isInteger(number)) {
            if (number >= -2147483648 && number <= 2147483647) {
                return 'INT';
            } else {
                return 'BIGINT';
            }
        } else {
            if (number >= -9999999 && number <= 9999999) {
                return 'DECIMAL(10, 2)';
            } else {
                return 'DECIMAL(20, 2)';
            }
        }
    } else if (value === 'true' || value === 'false') {
        return 'BOOLEAN';
    } else {
        if (value.length <= 255) {
            return 'VARCHAR(255)';
        } else {
            return 'TEXT';
        }
    }
}

function isChinese(text: string) {
    var re = /[\u4E00-\u9FA5\uF900-\uFA2D]/;
    return re.test(text);
}

export function handleUpload(files: File[]) {
    return new Promise(res => {
        const ddlArr: string[] = [];
        for (let i = 0; i < files.length; i++) {
            const file = files[i];
            const reader = new FileReader();
            reader.onload = (e: any) => {
                const data = new Uint8Array(e.target.result);
                const workbook = XLSX.read(data, {
                    type: 'array',
                    cellDates: true,
                    cellNF: true,
                });

                const sheetName = workbook.SheetNames[0];
                const jsonData = XLSX.utils.sheet_to_json<any>(workbook.Sheets[sheetName], {
                    defval: '',
                    header: 1,
                });
                let tableName = file.name.split('.')[0];
                if (isChinese(tableName)) {
                    tableName = flatten(
                        pinyin(tableName, {
                            style: 'normal',
                        })
                    ).join('_');
                }

                const columns: any[] = jsonData[0];
                const columnTypes = map(columns, (_, column) => {
                    const firstNonNullValue = get(
                        jsonData.slice(1).find(row => row[column] != null),
                        [column],
                        'VARCHAR(255)'
                    );
                    return getDataType(firstNonNullValue);
                });

                const ddl = `CREATE TABLE \`${tableName}\` (${map(
                    columns,
                    (column, i) => `\`${column}\` ${columnTypes[i]}`
                ).join(', ')});`;
                ddlArr.push(ddl);
                if (ddlArr.length === files.length) {
                    res(ddlArr);
                }
            };
            reader.readAsArrayBuffer((file as File & { originFile: File })?.originFile);
        }
    });
}

const ImportExcel = ({ setList }: { setList: typeof useState<UploadProps['fileList']> }) => {
    return (
        <Upload
            drag
            multiple
            accept=".xls,.xlsx,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
            onDrop={e => {
                let uploadFile = e.dataTransfer.files;
            }}
            autoUpload={false}
            onChange={lists => setList(lists)}
        />
    );
};

export default ImportExcel;
