import { get, map, uniqueId } from 'lodash';
import { useContext, useEffect, useMemo, useState } from 'react';
import { CustomProps, DagContextType } from '../context';
import {
  Button,
  Dropdown,
  Form,
  Input,
  Menu,
  Modal,
  Popover,
  Select,
  Trigger,
} from '@arco-design/web-react';
import { IconDelete, IconPlus } from '@arco-design/web-react/icon';
import { Graph, Node } from '@antv/x6';
import { DagreLayout } from '@antv/layout';

export interface NodeStatus {
  id: string;
  status: 'default' | 'success' | 'failed' | 'running';
  label?: string;
  nodeType?: NodeType;
}

export enum NodeType {
  'FLOW_START',
}

const image = {
  logo: 'https://gw.alipayobjects.com/mdn/rms_43231b/afts/img/A*evDjT5vjkX0AAAAAAAAAAAAAARQnAQ',
  success: 'https://gw.alipayobjects.com/mdn/rms_43231b/afts/img/A*6l60T6h8TTQAAAAAAAAAAAAAARQnAQ',
  failed: 'https://gw.alipayobjects.com/mdn/rms_43231b/afts/img/A*SEISQ6My-HoAAAAAAAAAAAAAARQnAQ',
  running: 'https://gw.alipayobjects.com/mdn/rms_43231b/afts/img/A*t8fURKfgSOgAAAAAAAAAAAAAARQnAQ',
};

function removeNodeAndConnected(node: Node, graph: Graph) {
  // 获取与节点关联的所有边
  const connectedEdges = graph.getConnectedEdges(node, { deep: true });

  connectedEdges.forEach(edge => {
    // 找出这个节点作为source的目标节点
    if (edge.getSourceCell() === node) {
      const target = edge.getTargetCell() as Node;
      if (target) {
        // 递归删除目标节点及其关联的节点和边
        removeNodeAndConnected(target, graph);
      }
    }

    // 删除边
    graph.removeCell(edge);
  });

  // 删除节点
  graph.removeCell(node);
}

const FormItem = Form.Item;
const Option = Select.Option;

const data = {
  函数: ['Chengdu', 'Mianyang', 'Aba'],
  'SQL 查询': ['Beijing', 'Shanghai', 'Tianjin'],
  算子: ["Xi'an", "Xi'an", "Xi'an"],
};

function MenuList({ node, graph, nodeType }: { node: Node; graph: Graph; nodeType: NodeType }) {
  const types = Object.keys(data) as Array<keyof typeof data>;
  const defaultType = types[0];
  const [type, setType] = useState<keyof typeof data>(defaultType);
  const [cities, setCities] = useState<string[]>([]);
  const [city, setCity] = useState('');

  useEffect(() => {
    const cities = data[type] || [];
    setCities(cities);
    setCity(cities[0]);
  }, [type]);

  return (
    <div className="w-[150px] min-h-[50px]">
      <Button
        type="default"
        className="cursor-pointer w-full text-left my-[5px]"
        icon={<IconPlus />}
        onClick={() => {
          Modal.confirm({
            title: '配置节点',
            icon: null,
            footer: null,
            content: (
              <div>
                <Form autoComplete="off">
                  <FormItem label="函数类型">
                    <Select
                      placeholder="Select Province"
                      onChange={value => setType(value)}
                      defaultValue={type}
                    >
                      {map(data, (_, k) => {
                        console.log(k, 'kkk');
                        return (
                          <Option key={k} value={k}>
                            {k}
                          </Option>
                        );
                      })}
                    </Select>
                  </FormItem>
                  <FormItem label="函数名称">
                    <Select
                      placeholder="Select city"
                      onChange={value => setCity(value)}
                      value={city}
                    ></Select>
                  </FormItem>
                  <FormItem wrapperCol={{ offset: 5 }}>
                    <Button type="primary">确定</Button>
                  </FormItem>
                </Form>
              </div>
            ),
          });
          const NeedPorts = [
            {
              id: node.id + 'port' + uniqueId(),
              group: 'bottom',
            },
            {
              id: node.id + 'port' + uniqueId(),
              group: 'top',
            },
          ];

          const childNode = {
            id: node.id + '_' + 'node' + uniqueId(),
            shape: 'dag-node',
            x: 0,
            y: 0,
            data: {
              label: '读数据',
              status: 'success',
            },
            ports: [NeedPorts[1]],
          };

          node.addPort(NeedPorts[0]);
          setTimeout(() => {
            graph.addNode(childNode);
            graph.addEdge({
              id: node.id + '_' + 'edge' + uniqueId(),
              shape: 'dag-edge',
              source: {
                cell: node.id,
                port: NeedPorts[0].id,
              },
              target: {
                cell: childNode.id,
                // port: NeedPorts[1].id,
              },
              zIndex: 0,
            });
            // 获取所有节点
            const nodes = graph
              .getNodes()
              .map(node => ({ id: node.id, raw: node, ...node.size() }));

            // 获取所有边
            const edges = graph.getEdges().map(edge => ({
              source: edge.getSourceCell()!.id,
              target: edge.getTargetCell()!.id,
              raw: edge,
            }));
            const dagreLayout = new DagreLayout({
              type: 'dagre',
              rankdir: 'TB',
              ranksep: 30,
              nodesep: 70,
            });
            const model = dagreLayout.layout({
              nodes,
              edges,
            }) as {
              nodes: ((typeof nodes)[0] & { x: number; y: number })[];
              edges: typeof edges;
            };

            (model.nodes || []).forEach(({ raw, x, y }) => {
              raw.position(x, y);
            });
            graph.centerContent();
          }, 50);
        }}
      >
        添加子节点
      </Button>
      <Button
        hidden={nodeType === NodeType.FLOW_START}
        type="default"
        status="danger"
        className="cursor-pointer w-full text-left my-[5px]"
        icon={<IconDelete />}
        onClick={() => {
          removeNodeAndConnected(node, graph);
        }}
      >
        删除
      </Button>
    </div>
  );
}

const AlgoNode = (props: CustomProps) => {
  const { node, graph } = props;
  const data = node?.getData() as NodeStatus;
  const { label, status = 'default', nodeType } = data;
  const DagContext: DagContextType = get(graph, 'DagContext')!;
  const { activeCell } = useContext(DagContext);
  graph.removeConnectedEdges;
  function render() {
    if (nodeType === NodeType.FLOW_START) {
      return (
        <Button type="outline" className="w-full" shape="round" status="success">
          流程开始
        </Button>
      );
    } else {
      return (
        <div className={`node ${status}`}>
          <img src={image.logo} alt="logo" />
          <span className="label">{label}</span>
          <span className="status">
            {status === 'success' && <img src={image.success} alt="success" />}
            {status === 'failed' && <img src={image.failed} alt="failed" />}
            {status === 'running' && <img src={image.running} alt="running" />}
          </span>
        </div>
      );
    }
  }
  return (
    <>
      {render()}
      <Popover
        trigger="hover"
        content={<MenuList node={node} graph={graph} nodeType={nodeType} />}
        position="right"
        title="操作"
      >
        <Button
          hidden={activeCell?.id !== node?.id}
          icon={<IconPlus />}
          shape="circle"
          iconOnly
          type="primary"
          className="inset-y-0 right-[-30px] m-auto absolute"
          size="mini"
        ></Button>
      </Popover>
    </>
  );
};

export default {
  shape: 'dag-node',
  width: 180,
  height: 36,
  component: AlgoNode,
  ports: {
    groups: {
      top: {
        position: 'top',
        attrs: {
          circle: {
            r: 4,
            magnet: true,
            stroke: '#C2C8D5',
            strokeWidth: 1,
            fill: '#fff',
          },
        },
      },
      bottom: {
        position: 'bottom',
        attrs: {
          circle: {
            r: 4,
            magnet: true,
            stroke: '#C2C8D5',
            strokeWidth: 1,
            fill: '#fff',
          },
        },
      },
    },
  },
};
