import {
    Alert,
    Button,
    Card,
    Collapse,
    Input,
    Select,
    Spin,
    Tag,
    Tooltip,
} from '@arco-design/web-react';
import {
    IconCode,
    IconEye,
    IconPlus,
    IconRefresh,
    IconSend,
    IconSwap,
} from '@arco-design/web-react/icon';
import React, {
    ReactElement,
    memo,
    useCallback,
    useEffect,
    useMemo,
    useRef,
    useState,
} from 'react';
import {
    LiveEditor,
    LivePreview,
    LiveProvider,
    useLiveContext,
    useRunner,
} from 'react-live-runner';
import * as styledComponents from 'styled-components';
import * as echarts from 'echarts';
import ReactDOM from 'react-dom';
import ReactECharts from 'echarts-for-react';
import AI from '@/components/AITool';
import useSWRMutation from 'swr/mutation';
import getView from '@/client/api/getView';
import { nanoid } from 'nanoid';
import { get, isEqual, map, set } from 'lodash';
import { useTranslation } from 'react-i18next';
import Editor, { Monaco, useMonaco, OnMount } from '@monaco-editor/react';
import { XML } from '@/utils/getXMLContent';
import { useWorker } from '@koale/useworker';
import dynamic from 'next/dynamic';
const ReactJson = dynamic(() => import('react-json-view'), { ssr: false });
import { Parser } from 'acorn';
import ConnectDb, { Query } from '@/client/api/connectDb';
import 'systemjs';
import { simple } from 'acorn-walk';
import jsx from 'acorn-jsx';
import i18n from '@/next-i18next.config';

const CollapseItem = Collapse.Item;

type MonacoEditor = Parameters<OnMount>[0];
declare const System: any;

function Error() {
    const { error } = useLiveContext();
    console.log(error, 'error');
    return <code>{error}</code>;
}

// const initText = i18n.getFixedT(i18n.language, 'chatView')('Function for processing data');

const init = (initText: string): string => {
    return `/**
 * @description ${initText}
 * @param  record<string,any> data
 */
`;
};

async function getUMDPath(packageName: string) {
    const response = await fetch(`https://data.jsdelivr.com/v1/packages/npm/${packageName}`);
    const packageJson = await response.json();
    const packageLatest = get(packageJson, 'versions[0]');
    const version = get(packageLatest, 'version');
    const entryFile = await (await fetch(get(packageLatest, 'links.entrypoints'))).json();
    // 一些包可能会在 "browser" 字段指定UMD模块的路径
    return map(get(entryFile, 'entrypoints', []), (item: any, type: string) => {
        return {
            type: type,
            url: `https://cdn.jsdelivr.net/npm/${packageName}@${version}${get(item, 'file')}`,
        };
    });
}

function loadUMDModule(packageName: string) {
    window.React = React;
    window.ReactDOM = ReactDOM;
    return new Promise((resolve, reject) => {
        try {
            const script = document.createElement('script');
            script.type = 'module';
            const initFnName: any = [packageName + 'Init'];
            window[initFnName] = mod => {
                resolve(mod);
                delete window[initFnName];
            };

            const errHandler = e => {
                resolve(loadUMDModuleBySystem(packageName));
                window.removeEventListener('error', errHandler);
                console.log('error: esm load err', e);
            };

            window.addEventListener('error', errHandler);

            script.textContent = `
                import * as module from "https://cdn.jsdelivr.net/npm/${packageName}/+esm";
                console.log(module,"module")
                window["${initFnName}"] && window["${initFnName}"](module)
            `;
            document.head.append(script);
            (async () => {
                const umdPath = await getUMDPath(packageName);
                for (const item of umdPath) {
                    if (item.type === 'css') {
                        System.import(item.url).then(function (module: any) {
                            const styleSheet = module.default; // A CSSStyleSheet object
                            console.log(styleSheet, "styleSheet")
                            document.adoptedStyleSheets = [...document.adoptedStyleSheets, styleSheet]; // now your css is available to be used.
                        });
                    }
                }
            })()
        } catch (e) {
            console.log(e);
        }
    });
}

async function loadUMDModuleBySystem(packageName: string) {
    try {
        window.define = undefined;
        window.React = React;
        window.ReactDOM = ReactDOM;
        const umdPath = await getUMDPath(packageName);
        let mod;
        for (const item of umdPath) {
            const res = await System.import(item.url);
            if (item.type === 'js') {
                mod = res;
            }
        }
        return mod;
    } catch (error) {
        console.error(error);
    }
}

function ReactView({ props }: { props: Record<string, any> }) {
    const { code } = useLiveContext();
    const { t } = useTranslation('chatView');

    return (
        <div>
            <div className="flex justify-end my-[5px]">
                <Button
                    className="mr-[60px] shadow"
                    type="primary"
                    size="small"
                    shape="round"
                    onClick={() => {
                        if (window.parent) {
                            window.parent.postMessage(
                                {
                                    code: code,
                                    type: 'code',
                                    props,
                                },
                                '*'
                            );
                        }
                    }}
                >
                    <IconPlus /> {t('Add component')}
                </Button>
            </div>
            <LivePreview className="overflow-auto  min-h-[350px]" />
        </div>
    );
}

export function ChatView({
    defaultNode,
    props: propsRaw,
    id,
    functions = '',
}: {
    defaultNode?: ReactElement;
    props: Record<string, any>;
    id: string;
    functions: Query['content']['functions'];
}) {
    const [showCode, setShowCode] = useState(true);
    const reqRef = useRef(nanoid());
    const [showTable, setShowTable] = useState(false);
    const [props, setProps] = useState(propsRaw);
    const { t } = useTranslation('chatView');
    const [dependenciesMap, setDepend] = useState({
        react: React,
        lodash: get(window, '_'),
        'react-dom': ReactDOM,
        'styled-components': styledComponents,
        'echarts-for-react': ReactECharts,
        echarts: {
            default: echarts,
            echarts: echarts,
            ...echarts,
        },
    });

    useEffect(() => {
        setProps(propsRaw);
    }, [propsRaw]);

    const [loadDependName, setLoadDependName] = useState<ReactElement | string>('');

    const { trigger, data, isMutating } = useSWRMutation(
        reqRef.current,
        (_, { arg: { need } }) => {
            return getView.getViewComponent({
                props,
                need,
            });
        },
        {
            async onSuccess(data) {
                // loadUMDModule('echarts').then(mod => {
                //     console.log(mod, 'mod');
                // });
                const example = get(data, 'data.code');
                const ast = Parser.extend(jsx()).parse(example, {
                    sourceType: 'module',
                    ecmaVersion: 6,
                });

                const dependencies: string[] = [];

                try {
                    simple(ast, {
                        ImportDeclaration(node: any) {
                            node.source?.value && dependencies.push(node.source.value);
                        },
                    });
                } catch (e) { }
                const deps: any = {};
                for (const depend of dependencies) {
                    if (!get(dependenciesMap, depend)) {
                        setLoadDependName(<div>
                            {t("Loading dependencies")}: <Tag bordered color="orange">{depend}</Tag>
                        </div>)
                        await loadUMDModule(depend).then(mod => {
                            console.log(mod);
                            if (mod) {
                                deps[depend] = mod;
                            }
                        });
                        setLoadDependName(<div>
                            {t("Loading completed")}: <Tag bordered color="orange">{depend}</Tag>
                        </div>)
                    }
                }
                setLoadDependName("")
                setDepend({
                    ...dependenciesMap,
                    ...deps,
                });
            },
        }
    );

    const SendButton = useCallback(
        ({ inputRef }: { inputRef: any }) => {
            return (
                <Button
                    shape="circle"
                    icon={data ? <IconRefresh /> : <IconSend />}
                    size="mini"
                    type="primary"
                    loading={isMutating}
                    onClick={async () => {
                        await trigger({ need: inputRef.current.dom.value });
                        setShowTable(false);
                    }}
                />
            );
        },
        [isMutating, trigger, setShowTable]
    );

    const example = get(data, 'data.code');
    functions = functions || init(t('Function for processing data'));

    const Live = useMemo(() => {
        let node;
        try {
            node = !showCode ? <LiveEditor /> : <ReactView props={props} />;
        } catch (err) {
            node = <LiveEditor />;
        }
        console.log(props, "props")

        return (
            <LiveProvider
                code={example}
                scope={{
                    data: props,
                    import: dependenciesMap,
                }}
            >
                {node}
                <Error />
            </LiveProvider>
        );
    }, [example, props, showCode, dependenciesMap]);

    const [editor, setEditor] = useState<MonacoEditor>();
    const monaco = useMonaco();
    const [workerFn, { status: workerStatus, kill: workerTerminate }] = useWorker(
        (data, code) => {
            return new Function('data', code)(data);
        },
        {
            autoTerminate: false,
        }
    );

    const onKeyDown = editor?.onKeyDown!;
    const KeyDownEvent = useRef<ReturnType<typeof onKeyDown>>();

    useEffect(() => {
        if (editor) {
            if (KeyDownEvent.current) {
                KeyDownEvent.current.dispose();
            }
            KeyDownEvent.current = editor.onKeyDown(function (event: any) {
                if (event.keyCode === 2 && monaco) {
                    const position = editor.getPosition();
                    const line = editor.getModel()?.getLineContent(position!.lineNumber) || '';
                    const flag = /^\/\//;
                    if (flag.test(line)) {
                        // 在当前行后面提示“生成中”占位符
                        const position = editor.getPosition()!;
                        const lineNumber = position.lineNumber;
                        const column = position.column;
                        const insertText = ` ` + t('Code generation in progress, please wait...');
                        const op = {
                            range: new monaco.Range(lineNumber, column, lineNumber, column),
                            text: insertText,
                        };
                        editor.executeEdits('insertSnippet', [op]);

                        getView
                            .getViewFunction({
                                data: props,
                                need: line.replace(flag, ''),
                            })
                            .then((data: any) => {
                                const code = get(data, 'data.code');
                                const xml = new XML(code);
                                const insertText = xml.get('FunctionCode');

                                const nextLineNumber = lineNumber + 1;
                                const op1 = {
                                    range: new monaco.Range(
                                        lineNumber,
                                        0,
                                        lineNumber,
                                        column + insertText.length
                                    ),
                                    text: line,
                                };
                                const op2 = {
                                    range: new monaco.Range(
                                        lineNumber,
                                        column + insertText.length,
                                        nextLineNumber,
                                        1
                                    ),
                                    text: '\n' + insertText.trim() + '\n\n',
                                };

                                editor.executeEdits('insertSnippet', [op1, op2]);
                            });
                    }
                }
            });
        }
    }, [editor, props]);

    const [functionName, setFunctionName] = useState('');
    const [FunctionOption, setFunctionOption] = useState<ReturnType<typeof Parser.parse>[]>([]);
    return (
        <div>
            {propsRaw.length ? (
                <>
                    <div className="flex justify-between items-center mb-[20px]">
                        <AI
                            simpleMode="input"
                            startView={3}
                            inputProps={{
                                style: { width: 400 },
                                height: 36,
                                className: 'overflow-hidden simple-mode-border',
                                prefix: (
                                    <span className="text-[var(--pc)]">
                                        {t('component display')}
                                    </span>
                                ),
                                autoFocus: false,
                            }}
                            SendButton={SendButton}
                            messageList={[]}
                            setMessageList={function (value: any[]) {
                                return false;
                            }}
                        />
                        {data && (
                            <Button shape="circle" onClick={() => setShowTable(!showTable)}>
                                <IconSwap />
                            </Button>
                        )}
                    </div>
                    <div className="w-full mb-[20px]">
                        <Collapse>
                            <CollapseItem
                                header={
                                    <div>
                                        {t('Data processing')}
                                        <span className="text-[var(--pc)] mx-[10px] text-[12px]">
                                            ({t('Comment + Tab to generate function')})
                                        </span>
                                    </div>
                                }
                                name="1"
                                extra={
                                    <div className="flex">
                                        <Select
                                            placeholder={t('Execute function name')}
                                            size="mini"
                                            allowClear
                                            style={{
                                                width: '200px',
                                            }}
                                            showSearch={{
                                                retainInputValue: true,
                                            }}
                                            onClick={() => {
                                                let comments: { [line: string]: string } = {};
                                                const root: any = Parser.parse(
                                                    editor?.getValue() || functions,
                                                    {
                                                        locations: true,
                                                        ranges: true,
                                                        sourceType: 'module',
                                                        ecmaVersion: 2020,
                                                        onComment: (...arg) => {
                                                            if (arg[1]) {
                                                                comments[arg[3] + 1] = arg[1];
                                                            }
                                                        },
                                                    }
                                                );

                                                const body: ReturnType<typeof Parser.parse>[] = get(
                                                    root,
                                                    'body'
                                                );

                                                setFunctionOption(
                                                    body.filter(v => {
                                                        set(v, 'comment', comments[v.start]);
                                                        return v.type === 'FunctionDeclaration';
                                                    })
                                                );
                                            }}
                                            onChange={setFunctionName}
                                        >
                                            {FunctionOption.map(item => {
                                                const name = get(item, 'id.name', '');
                                                return (
                                                    <Select.Option key={name} value={name}>
                                                        <Tooltip content={get(item, 'comment')}>
                                                            <span className="text-[12px] leading-[24px] text-[var(--pc)] mr-[5px]">
                                                                {get(
                                                                    item,
                                                                    'comment',
                                                                    t(
                                                                        'No function description available'
                                                                    )
                                                                )}
                                                            </span>
                                                        </Tooltip>
                                                        <div>{name}</div>
                                                    </Select.Option>
                                                );
                                            })}
                                        </Select>
                                        <Button
                                            className="ml-[10px]"
                                            onClick={async () => {
                                                if (!functionName) {
                                                    setProps(propsRaw);
                                                    return;
                                                }
                                                const data = await workerFn(
                                                    propsRaw,
                                                    `${editor?.getValue() || functions
                                                    };\n return ${functionName}(data);`
                                                );
                                                ConnectDb.updateQuery(
                                                    id,
                                                    `${editor?.getValue() || functions}\n`
                                                );
                                                setProps(data);
                                            }}
                                            type="primary"
                                            size="mini"
                                            shape="round"
                                        >
                                            {t('Run')}
                                        </Button>
                                    </div>
                                }
                            >
                                <Editor
                                    onMount={(instance, monaco) => {
                                        console.log(instance);
                                        setEditor(instance);
                                    }}
                                    height="300px"
                                    defaultLanguage="javascript"
                                    defaultValue={functions}
                                />
                            </CollapseItem>
                        </Collapse>
                    </div>
                </>
            ) : (
                <Alert
                    style={{ marginBottom: 20 }}
                    type="success"
                    content={t('Execution successful')}
                />
            )}
            <Spin loading={isMutating} block className="my-[20px]">
                {example && !showTable ? (
                    <div>
                        <div className="text-[20px] absolute text-orange-200 z-[99] right-[20px]">
                            {showCode ? (
                                <IconCode onClick={() => setShowCode(false)} />
                            ) : (
                                <IconEye onClick={() => setShowCode(true)} />
                            )}
                        </div>
                        <div className="overflow-hidden">{loadDependName ? <div className="flex justify-center">
                            <Spin block loading />
                            <div className="ml-[10px]"> {loadDependName}</div>
                        </div> : Live}</div>
                    </div>
                ) : isEqual(props, propsRaw) ? (
                    defaultNode || null
                ) : (
                    <ReactJson src={props} />
                )}
            </Spin>
        </div>
    );
}

export default function View() {
    return null;
}
